// Copyright (c) FIRST and other WPILib contributors.
// Copyright (c) 5813 Morpheus, 2023
// Open Source Software; you can modify and/or share it under the terms of
// the WPILib BSD license file in the root directory of this project.

package team.frc.morpheus.chargedup.subsystems;

import com.ctre.phoenix.ErrorCode;
import com.ctre.phoenix.motorcontrol.TalonFXControlMode;
import com.ctre.phoenix.motorcontrol.TalonFXFeedbackDevice;
import com.ctre.phoenix.motorcontrol.TalonSRXControlMode;
import com.ctre.phoenix.motorcontrol.TalonSRXFeedbackDevice;
import com.ctre.phoenix.motorcontrol.can.TalonFX;
import com.ctre.phoenix.motorcontrol.can.TalonSRX;
import com.ctre.phoenix.motorcontrol.can.TalonSRXConfiguration;
import edu.wpi.first.math.geometry.Rotation2d;
import edu.wpi.first.math.kinematics.SwerveModulePosition;
import edu.wpi.first.math.kinematics.SwerveModuleState;
import edu.wpi.first.wpilibj.Timer;
import team.frc.morpheus.chargedup.Constants.ModuleConstants;
import team.frc.morpheus.chargedup.utils.CTREModuleState;

public class SwerveModule {
  private final TalonFX m_driveMotor;
  private final TalonSRX m_turningMotor;
  private final int absoluteOrigin;
  private boolean seeded = false;

  /**
   * Constructs a SwerveModule.
   *
   * @param driveMotorChannel The CAN address of the drive motor.
   * @param turningMotorChannel The CAN address of the turning motor.
   * @param turningEncoderReversed Whether the turning encoder is reversed.
   * @param absoluteOrigin The zero measurment from absolute mode.
   */
  public SwerveModule(
      int driveMotorChannel,
      int turningMotorChannel,
      boolean turningEncoderReversed,
      int absoluteOrigin) {
    m_driveMotor = new TalonFX(driveMotorChannel);
    m_turningMotor = new TalonSRX(turningMotorChannel);

    // Clear out any existing persistent state
    m_driveMotor.configFactoryDefault();
    m_driveMotor.configSelectedFeedbackSensor(
            TalonFXFeedbackDevice.IntegratedSensor, 0, 0);
    m_driveMotor.config_kF(0, ModuleConstants.kFModuleDriveController);
    m_driveMotor.config_kP(0, ModuleConstants.kPModuleDriveController);
    m_driveMotor.configPeakOutputForward(1.0);
    m_driveMotor.configPeakOutputReverse(-1.0);
    m_driveMotor.configNominalOutputForward(0.0);
    m_driveMotor.configNominalOutputReverse(0.0);

    final TalonSRXConfiguration turningConfiguration = new TalonSRXConfiguration();
    turningConfiguration.primaryPID.selectedFeedbackSensor =
        TalonSRXFeedbackDevice.CTRE_MagEncoder_Relative.toFeedbackDevice();
    turningConfiguration.auxiliaryPID.selectedFeedbackSensor =
        TalonSRXFeedbackDevice.CTRE_MagEncoder_Absolute.toFeedbackDevice();
    turningConfiguration.slot0.kF = ModuleConstants.kFModuleTurningController;
    turningConfiguration.slot0.kP = ModuleConstants.kPModuleTurningController;
    turningConfiguration.peakOutputForward = 1.0;
    turningConfiguration.peakOutputReverse = -1.0;
    turningConfiguration.nominalOutputForward = 0.0;
    turningConfiguration.nominalOutputReverse = 0.0;
    turningConfiguration.motionCruiseVelocity = toNativeUnits(
        ModuleConstants.kMaxModuleAngularSpeedRadiansPerSecond,
        ModuleConstants.kTurningEncoderDistancePerPulse);
    turningConfiguration.motionAcceleration = toNativeUnits(
        ModuleConstants.kMaxModuleAngularAccelerationRadiansPerSecondSquared,
        ModuleConstants.kTurningEncoderDistancePerPulse);
    ErrorCode e;
    while((e = m_turningMotor.configAllSettings(turningConfiguration, 1000)) != ErrorCode.OK)
      System.out.println("SwerveModule startup: config all settings failed with error " + e);
    m_turningMotor.setInverted(true);
    // Set whether turning encoder should be reversed or not
    m_turningMotor.setSensorPhase(turningEncoderReversed);

    this.absoluteOrigin = absoluteOrigin;
  }

  /**
   * Given a velocity in some unit per second and the scale factor indicating the number of these
   * units per sensor unit, the corresponding velocity in Talon native units is calculated and
   * returned.
   * @param unitsPerSecond A velocity in some unit per second.
   * @param unitsPerPulse The scale factor between sensor units and "some" unit.
   * @return The velocity in Talon native units.
   */
  private static double toNativeUnits(double unitsPerSecond, double unitsPerPulse) {
    // Talon native units for velocity are sensor units (encoder counts) per 100 milliseconds.
    return (unitsPerSecond / unitsPerPulse) / 10.0;
  }

  /**
   * Returns the distance traveled by a point on the circumference of the wheel, in meters.
   *
   * @return The distance traveled by a point on the circumference of the wheel, in meters.
   */
  private double getDriveDistance() {
    return m_driveMotor.getSelectedSensorPosition() *
            ModuleConstants.kDriveEncoderDistancePerPulse;
  }

  /**
   * Returns the current velocity of the surface of the wheel, in meters per second.
   *
   * @return The current velocity of the surface of the wheel, in meters per second.
   */
  private double getDriveVelocity() {
    return m_driveMotor.getSelectedSensorVelocity() *
            ModuleConstants.kDriveEncoderDistancePerPulse;
  }

  /**
   * Returns the current angle of the module, in radians.
   *
   * @return The current angle of the module, in radians.
   */
  private double getAngle() {
    return m_turningMotor.getSelectedSensorPosition() *
            ModuleConstants.kTurningEncoderDistancePerPulse;
  }

  /**
   * Returns the current state of the module.
   *
   * @return The current state of the module.
   */
  public SwerveModuleState getState() {
    return new SwerveModuleState(getDriveVelocity(), new Rotation2d(getAngle()));
  }

  /**
   * Returns the current position of the module.
   *
   * @return The current position of the module.
   */
  public SwerveModulePosition getPosition() {
    return new SwerveModulePosition(getDriveDistance(), new Rotation2d(getAngle()));
  }

  /**
   * Sets the desired state for the module.
   *
   * @param desiredState Desired state with speed and angle.
   */
  public void setDesiredState(SwerveModuleState desiredState) {
    if (!seeded) absoluteSeedRelative();

    // Optimize the reference state to avoid spinning further than 90 degrees
    SwerveModuleState state =
        CTREModuleState.optimize(desiredState, new Rotation2d(getAngle()));

    m_driveMotor.set(TalonFXControlMode.Velocity, toNativeUnits(
            state.speedMetersPerSecond,
            ModuleConstants.kDriveEncoderDistancePerPulse));
    m_turningMotor.set(TalonSRXControlMode.MotionMagic,
            state.angle.getRadians() / ModuleConstants.kTurningEncoderDistancePerPulse);
  }

  /** Zeroes all the SwerveModule encoders. */
  public void resetEncoders() {
    m_driveMotor.setSelectedSensorPosition(0);
    m_turningMotor.setSelectedSensorPosition(0);
  }

  /**
   * Seeds the position of the relative turning encoder using the absolute encoder's position
   * relative to the absolute origin.
   */
  public void absoluteSeedRelative() {
    for (int i = 0; i < 3; i++) {
      if (i > 0) {
        System.out.println("SwerveModule startup: absoluteCurrent exactly 0.0, retrying...");
        Timer.delay(0.05);
      }

      final double absoluteCurrent = m_turningMotor.getSelectedSensorPosition(1);
      System.out.printf("SwerveModule startup: absoluteOrigin=%d, absoluteCurrent=%f\n",
          absoluteOrigin, absoluteCurrent);
      m_turningMotor.setSelectedSensorPosition(absoluteCurrent - absoluteOrigin, 0, 0);

      if (absoluteCurrent != 0.0) break;
    }
    seeded = true;
  }
}
