// Copyright (c) FIRST and other WPILib contributors.
// Copyright (c) 5813 Morpheus, 2023
// Open Source Software; you can modify and/or share it under the terms of
// the WPILib BSD license file in the root directory of this project.

package team.frc.morpheus.chargedup.subsystems;

import com.ctre.phoenix.sensors.WPI_Pigeon2;
import com.pathplanner.lib.auto.AutoBuilder;
import edu.wpi.first.math.geometry.Pose2d;
import edu.wpi.first.math.kinematics.*;
import edu.wpi.first.wpilibj.smartdashboard.SmartDashboard;
import edu.wpi.first.wpilibj2.command.SubsystemBase;
import team.frc.morpheus.chargedup.Constants.DriveConstants;

public class DriveSubsystem extends SubsystemBase {
  private static DriveSubsystem instance;
  // Robot swerve modules
  private final SwerveModule m_frontLeft =
      new SwerveModule(
          DriveConstants.kFrontLeftDriveMotorID,
          DriveConstants.kFrontLeftTurningMotorID,
          DriveConstants.kFrontLeftTurningEncoderReversed,
          DriveConstants.kFrontLeftAbsoluteOrigin);

  private final SwerveModule m_rearLeft =
      new SwerveModule(
          DriveConstants.kRearLeftDriveMotorID,
          DriveConstants.kRearLeftTurningMotorID,
          DriveConstants.kRearLeftTurningEncoderReversed,
          DriveConstants.kRearLeftAbsoluteOrigin);

  private final SwerveModule m_frontRight =
      new SwerveModule(
          DriveConstants.kFrontRightDriveMotorID,
          DriveConstants.kFrontRightTurningMotorID,
          DriveConstants.kFrontRightTurningEncoderReversed,
          DriveConstants.kFrontRightAbsoluteOrigin);

  private final SwerveModule m_rearRight =
      new SwerveModule(
          DriveConstants.kRearRightDriveMotorID,
          DriveConstants.kRearRightTurningMotorID,
          DriveConstants.kRearRightTurningEncoderReversed,
          DriveConstants.kRearRightAbsoluteOrigin);

  // The gyro sensor
  private final WPI_Pigeon2 m_gyro = new WPI_Pigeon2(DriveConstants.kGyroID);
  // Odometry class for tracking robot pose
  SwerveDriveOdometry m_odometry =
      new SwerveDriveOdometry(
          DriveConstants.kDriveKinematics,
          m_gyro.getRotation2d(),
          new SwerveModulePosition[] {
            m_frontLeft.getPosition(),
            m_frontRight.getPosition(),
            m_rearLeft.getPosition(),
            m_rearRight.getPosition()
          });

  /** Creates a new DriveSubsystem. */
  private DriveSubsystem() {
    instance = this;

    AutoBuilder.configureHolonomic(
            this::getPose,
            this::resetOdometry,
            () -> DriveConstants.kDriveKinematics.toChassisSpeeds(
                    m_frontLeft.getState(),
                    m_frontRight.getState(),
                    m_rearLeft.getState(),
                    m_rearRight.getState()),
            chassisSpeeds -> drive(
                    chassisSpeeds.vxMetersPerSecond,
                    chassisSpeeds.vyMetersPerSecond,
                    chassisSpeeds.omegaRadiansPerSecond,
                    false),
            DriveConstants.kPathFollowerConfig,
            this
    );
  }

  public static synchronized DriveSubsystem getInstance() {
    return instance != null ? instance : new DriveSubsystem();
  }

  @Override
  public void periodic() {
    // Update the odometry in the periodic block
    m_odometry.update(
        m_gyro.getRotation2d(),
        new SwerveModulePosition[] {
          m_frontLeft.getPosition(),
          m_frontRight.getPosition(),
          m_rearLeft.getPosition(),
          m_rearRight.getPosition()
        });
    SmartDashboard.putNumber("Gyro angle", m_gyro.getAngle());
  }

  /**
   * Returns the currently-estimated pose of the robot.
   *
   * @return The pose
   */
  public Pose2d getPose() {
    return m_odometry.getPoseMeters();
  }

  /**
   * Resets the odometry to the specified pose.
   *
   * @param pose The pose to which to set the odometry.
   */
  public void resetOdometry(Pose2d pose) {
    m_odometry.resetPosition(
        m_gyro.getRotation2d(),
        new SwerveModulePosition[] {
          m_frontLeft.getPosition(),
          m_frontRight.getPosition(),
          m_rearLeft.getPosition(),
          m_rearRight.getPosition()
        },
        pose);
  }

  /**
   * Method to drive the robot using joystick info.
   *
   * @param xSpeed Speed of the robot in the x direction (forward).
   * @param ySpeed Speed of the robot in the y direction (sideways).
   * @param rot Angular rate of the robot.
   * @param fieldRelative Whether the provided x and y speeds are relative to the field.
   */
  public void drive(double xSpeed, double ySpeed, double rot, boolean fieldRelative) {
    var swerveModuleStates =
        DriveConstants.kDriveKinematics.toSwerveModuleStates(
            fieldRelative
                ? ChassisSpeeds.fromFieldRelativeSpeeds(xSpeed, ySpeed, rot, m_gyro.getRotation2d())
                : new ChassisSpeeds(xSpeed, ySpeed, rot));
    SwerveDriveKinematics.desaturateWheelSpeeds(
        swerveModuleStates, DriveConstants.kMaxSpeedMetersPerSecond);
    m_frontLeft.setDesiredState(swerveModuleStates[0]);
    m_frontRight.setDesiredState(swerveModuleStates[1]);
    m_rearLeft.setDesiredState(swerveModuleStates[2]);
    m_rearRight.setDesiredState(swerveModuleStates[3]);
  }

  /**
   * Sets the swerve ModuleStates.
   *
   * @param frontLeft SwerveModuleState for front-left SwerveModule
   * @param frontRight SwerveModuleState for front-right SwerveModule
   * @param rearLeft SwerveModuleState for rear-left SwerveModule
   * @param rearRight SwerveModuleState for rear-right SwerveModule
   */
  public void setModuleStates(SwerveModuleState frontLeft,
                              SwerveModuleState frontRight,
                              SwerveModuleState rearLeft,
                              SwerveModuleState rearRight) {
    final SwerveModuleState[] desiredStates = {
            frontLeft, frontRight, rearLeft, rearRight
    };
    setModuleStates(desiredStates);
  }

  /**
   * Sets the swerve ModuleStates.
   *
   * @param desiredStates The desired states in the following order:
   *                      front-left, front-right, rear-left, and rear-right
   */
  public void setModuleStates(SwerveModuleState[] desiredStates) {
    SwerveDriveKinematics.desaturateWheelSpeeds(desiredStates,
        DriveConstants.kMaxSpeedMetersPerSecond);
    m_frontLeft.setDesiredState(desiredStates[0]);
    m_frontRight.setDesiredState(desiredStates[1]);
    m_rearLeft.setDesiredState(desiredStates[2]);
    m_rearRight.setDesiredState(desiredStates[3]);
  }

  /** Resets the drive encoders to currently read a position of 0. */
  public void resetEncoders() {
    m_frontLeft.resetEncoders();
    m_rearLeft.resetEncoders();
    m_frontRight.resetEncoders();
    m_rearRight.resetEncoders();
  }

  /**
   * Seeds the position of the relative turning encoders using each absolute encoder's position
   * relative to its corresponding absolute origin.
   */
  public void absoluteSeedRelative() {
    m_frontLeft.absoluteSeedRelative();
    m_rearLeft.absoluteSeedRelative();
    m_frontRight.absoluteSeedRelative();
    m_rearRight.absoluteSeedRelative();
  }

  /** Zeroes the heading of the robot. */
  public void zeroHeading() {
    m_gyro.reset();
  }

  /**
   * Returns the heading of the robot.
   *
   * @return the robot's heading in degrees, from -180 to 180
   */
  public double getHeading() {
    return m_gyro.getRotation2d().getDegrees();
  }

  public double getPitch() {
    return m_gyro.getPitch();
  }

  public double getRoll() {
    return m_gyro.getRoll();
  }

  /**
   * Returns the turn rate of the robot.
   *
   * @return The turn rate of the robot, in degrees per second
   */
  public double getTurnRate() {
    return m_gyro.getRate() * (DriveConstants.kGyroReversed ? -1.0 : 1.0);
  }
}
