package team.frc.morpheus.chargedup.commands.drivebase;

import java.util.ArrayList;
import java.util.function.BooleanSupplier;
import java.util.function.DoubleSupplier;

import edu.wpi.first.wpilibj2.command.Command;
import team.frc.morpheus.chargedup.Constants.DriveConstants;
import team.frc.morpheus.chargedup.subsystems.DriveSubsystem;
import team.frc.morpheus.chargedup.utils.CTREModuleState;
import edu.wpi.first.math.controller.PIDController;

public class OrdinalDrive extends Command {

    public static class OrdinalBinding{
        private BooleanSupplier requested;
        private double angle;

        public OrdinalBinding(BooleanSupplier requested, double angle) {
            this.requested = requested;
            this.angle = angle;
        }

        public boolean requested(){
            return requested.getAsBoolean();
        }

        public double angle(){
            return angle;
        }
    }
    
    private final DoubleSupplier xThrottle;
    private final DoubleSupplier yThrottle;
    private final DoubleSupplier steering;
    private final ArrayList<OrdinalBinding> bindings = new ArrayList<OrdinalBinding>(4);
    private final DriveSubsystem drive = DriveSubsystem.getInstance();
    private final PIDController pid = new PIDController(0.2, 0, 0);
    private boolean autoSteer;

    public OrdinalDrive(DoubleSupplier xThrottle, DoubleSupplier yThrottle, DoubleSupplier steering) {
        this.xThrottle = xThrottle;
        this.yThrottle = yThrottle;
        this.steering = steering;

        addRequirements(drive);
    }

    public OrdinalDrive addBinding(OrdinalBinding binding){
        bindings.add(binding);
        return this;
    }

    @Override
    public void initialize() {
        
    }

    private static double constrain(double value, double magnitudeAllowed) {
        final double sign = value >= 0 ? 1.0 : -1.0;
        final double absValue = Math.abs(value);

        return absValue > magnitudeAllowed ? sign * magnitudeAllowed : value;
    }

    @Override
    public void execute() {
        double steer = steering.getAsDouble();
        final double heading = drive.getHeading();

        for (OrdinalBinding binding : bindings) {
            if (binding.requested()) {
                // ToDo: Find closest real angle for setpoint
                pid.setSetpoint(CTREModuleState.placeInAppropriate0To360Scope(heading, binding.angle()));
                autoSteer = true;
                pid.reset();
            }
        }

        if (autoSteer){
            if (Math.abs(steer) > 0.10) autoSteer = false;
            else {
                steer = constrain(pid.calculate(heading), Math.PI);
            }
        } 

        drive.drive(
            xThrottle.getAsDouble() * DriveConstants.kMaxSpeedMetersPerSecond,
            yThrottle.getAsDouble() * DriveConstants.kMaxSpeedMetersPerSecond,
            steer,
            true
        );
    }

    @Override
    public boolean isFinished() {
        return true;
    }

    @Override
    public void end(boolean interrupted) {

    }
}
