// Copyright (c) FIRST and other WPILib contributors.
// Open Source Software; you can modify and/or share it under the terms of
// the WPILib BSD license file in the root directory of this project.

package team.frc.morpheus.chargedup;

import edu.wpi.first.wpilibj2.command.Commands;
import edu.wpi.first.wpilibj2.command.WaitCommand;
import team.frc.morpheus.chargedup.Constants.DriveConstants;
import team.frc.morpheus.chargedup.Constants.OperatorConstants;
import team.frc.morpheus.chargedup.commands.nado.NadoSet;
import team.frc.morpheus.chargedup.commands.nado.NadoState;
import team.frc.morpheus.chargedup.commands.nado.NadoSpinSet;
import team.frc.morpheus.chargedup.commands.nado.NadoWristAngle;
import team.frc.morpheus.chargedup.commands.Autos;
import team.frc.morpheus.chargedup.commands.drivebase.AutoBalance;
import team.frc.morpheus.chargedup.commands.drivebase.OrdinalDrive;
import team.frc.morpheus.chargedup.commands.drivebase.OrdinalDrive.OrdinalBinding;
import team.frc.morpheus.chargedup.subsystems.DriveSubsystem;
import team.frc.morpheus.chargedup.subsystems.Nado;
import edu.wpi.first.wpilibj.GenericHID.RumbleType;
import edu.wpi.first.wpilibj.smartdashboard.SendableChooser;
import edu.wpi.first.wpilibj.smartdashboard.SmartDashboard;
import edu.wpi.first.wpilibj2.command.Command;
import edu.wpi.first.wpilibj2.command.button.CommandXboxController;
import edu.wpi.first.wpilibj2.command.button.Trigger;

import java.util.HashMap;

/**
 * This class is where the bulk of the robot should be declared. Since Command-based is a
 * "declarative" paradigm, very little robot logic should actually be handled in the {@link Robot}
 * periodic methods (other than the scheduler calls). Instead, the structure of the robot (including
 * subsystems, commands, and trigger mappings) should be declared here.
 */
public class RobotContainer {
  // The robot's subsystems and commands are defined here...
  private final DriveSubsystem drive = DriveSubsystem.getInstance();
  private final Nado nado = Nado.getInstance();

  // Replace with CommandPS4Controller or CommandJoystick if needed
  private final CommandXboxController m_driverController =
      new CommandXboxController(OperatorConstants.kDriverControllerPort);

  private final SendableChooser<Command> autoChooser = new SendableChooser<>();

  /** The container for the robot. Contains subsystems, OI devices, and commands. */
  public RobotContainer() {
    // Configure the trigger bindings
    configureBindings();

    // Configure default commands

    // The left stick controls translation of the robot.
    // Turning is currently controlled by the X axis of the right stick.
    drive.setDefaultCommand(new OrdinalDrive(m_driverController::getLeftY, 
                                             m_driverController::getLeftX, 
                                             () -> { double x = m_driverController.getRightX();
                                                    return Math.abs(x) < 0.1 ? 0 : x * -0.5 * DriveConstants.kMaxAngularSpeedRadiansPerSecond;
                                              })
    .addBinding(new OrdinalBinding(m_driverController.povDown(), 0))
    .addBinding(new OrdinalBinding(m_driverController.povUp(), 180))
    .addBinding(new OrdinalBinding(m_driverController.povRight(), 90))
    .addBinding(new OrdinalBinding(m_driverController.povLeft(), -90))
    .addBinding(new OrdinalBinding(m_driverController.leftTrigger(0.1), 0))
    .addBinding(new OrdinalBinding(m_driverController.rightTrigger(0.1), 180)));

    configureAutonomousModes();
  }

  private void configureAutonomousModes() {
    autoChooser.setDefaultOption("No-op Auto", Commands.none());
    autoChooser.addOption("ShootHigh", Autos.shootHigh(nado));
    autoChooser.addOption("ShootLow", Autos.shootLow(nado));
    autoChooser.addOption("ShootHighCross", Autos.shootHighCross(nado, drive));
    autoChooser.addOption("ShootMid", Autos.shootMid(nado));
    autoChooser.addOption("ShootMidCross", Autos.shootMidCross(nado, drive));
    autoChooser.addOption("ShootMidBalance", Autos.shootMidBalance(nado, drive));
    autoChooser.addOption("Turn", Autos.turn(drive));
    autoChooser.addOption("ShootHighBalance", Autos.shootHighBalance(nado, drive));
    autoChooser.addOption("ShootHighCrossBalance", Autos.shootHighCrossBalance(nado, drive));

    SmartDashboard.putData("Autonomous Chooser", autoChooser);
  }

  /**
   * Use this method to define your trigger->command mappings. Triggers can be created via the
   * {@link Trigger#Trigger(java.util.function.BooleanSupplier)} constructor with an arbitrary
   * predicate, or via the named factories in {@link
   * edu.wpi.first.wpilibj2.command.button.CommandGenericHID}'s subclasses for {@link
   * CommandXboxController Xbox}/{@link edu.wpi.first.wpilibj2.command.button.CommandPS4Controller
   * PS4} controllers or {@link edu.wpi.first.wpilibj2.command.button.CommandJoystick Flight
   * joysticks}.
   */
  private void configureBindings() {
    // Allow the driver to reseed the relative encoder positions at runtime.
    m_driverController.leftStick().onTrue(Commands.runOnce(drive::absoluteSeedRelative, drive));

    //Intake
    //Sets the wrist to the right angle, and after it finishes sets the intake speed.
    //NadoSpinSet only ends if the intake current spikes over 30 amps, meaning the game piece
    //has been picked up and the wrist is put back into stow.
    m_driverController.rightBumper().onTrue(
      new NadoSet(NadoState.INTAKE).andThen(
      new NadoSpinSet(NadoState.INTAKE.spin, true),
      Commands.runOnce(() -> m_driverController.getHID().setRumble(RumbleType.kBothRumble,1)),
      new NadoSet(NadoState.STOW),
      Commands.runOnce(() -> m_driverController.getHID().setRumble(RumbleType.kBothRumble,0))));
    
    //Cancel button for the intake
    m_driverController.a().onTrue(new NadoSet(NadoState.STOW));

    //Shoot low
    m_driverController.b().onTrue(Commands.sequence(
      new NadoWristAngle(NadoState.SHOOT_LOW.wristAngle, NadoState.SHOOT_LOW.wristTolerance).raceWith(new WaitCommand(1.0)),
      new NadoSpinSet(NadoState.SHOOT_LOW.spin, false)));
    
    //Shoot low FAR
    m_driverController.leftBumper().onTrue(Commands.sequence(
      new NadoWristAngle(NadoState.SHOOT_LOW_FAR.wristAngle, NadoState.SHOOT_LOW_FAR.wristTolerance),
      new NadoSpinSet(NadoState.SHOOT_LOW_FAR.spin, false)));

    //Shoot mid
    m_driverController.x().onTrue(Commands.sequence(
      new NadoWristAngle(NadoState.SHOOT_MID.wristAngle, NadoState.SHOOT_MID.wristTolerance),
      new NadoSpinSet(NadoState.SHOOT_MID.spin, false)));

    //Shoot high
    m_driverController.y().onTrue(Commands.sequence(
      new NadoWristAngle(NadoState.SHOOT_HIGH.wristAngle, NadoState.SHOOT_HIGH.wristTolerance),
      new NadoSpinSet(NadoState.SHOOT_HIGH.spin, false)));

    m_driverController.start().whileTrue(new AutoBalance());
  }

  /**
   * Use this to pass the autonomous command to the main {@link Robot} class.
   *
   * @return the command to run in autonomous
   */
  public Command getAutonomousCommand() {
    // Auto retrieved from chooser
    return autoChooser.getSelected();
  }
}
