// Copyright (c) FIRST and other WPILib contributors.
// Open Source Software; you can modify and/or share it under the terms of
// the WPILib BSD license file in the root directory of this project.

package team.frc.morpheus.chargedup;

import com.pathplanner.lib.util.HolonomicPathFollowerConfig;
import com.pathplanner.lib.util.PIDConstants;
import com.pathplanner.lib.util.ReplanningConfig;
import edu.wpi.first.math.geometry.Translation2d;
import edu.wpi.first.math.kinematics.SwerveDriveKinematics;

/**
 * The Constants class provides a convenient place for teams to hold robot-wide numerical or boolean
 * constants. This class should not be used for any other purpose. All constants should be declared
 * globally (i.e. public static). Do not put anything functional in this class.
 *
 * <p>It is advised to statically import this class (or one of its inner classes) wherever the
 * constants are needed, to reduce verbosity.
 */
public final class Constants {
  public static final class DriveConstants {
    public static final int kFrontLeftDriveMotorID = 1;     // ID on roboRIO CAN bus
    public static final int kRearLeftDriveMotorID = 3;      // ID on roboRIO CAN bus
    public static final int kFrontRightDriveMotorID = 5;    // ID on roboRIO CAN bus
    public static final int kRearRightDriveMotorID = 7;     // ID on roboRIO CAN bus

    public static final int kFrontLeftTurningMotorID = 2;   // ID on roboRIO CAN bus
    public static final int kRearLeftTurningMotorID = 4;    // ID on roboRIO CAN bus
    public static final int kFrontRightTurningMotorID = 6;  // ID on roboRIO CAN bus
    public static final int kRearRightTurningMotorID = 8;   // ID on roboRIO CAN bus

    public static final int kGyroID = 9;                    // ID on roboRIO CAN bus

    public static final boolean kFrontLeftTurningEncoderReversed = true;
    public static final boolean kRearLeftTurningEncoderReversed = true;
    public static final boolean kFrontRightTurningEncoderReversed = true;
    public static final boolean kRearRightTurningEncoderReversed = true;

    public static final int kFrontLeftAbsoluteOrigin = 3931;
    public static final int kRearLeftAbsoluteOrigin = 2836;
    public static final int kFrontRightAbsoluteOrigin = 883;
    public static final int kRearRightAbsoluteOrigin = 191;

    public static final double kTrackWidth = 0.48895;
    // Distance between centers of right and left wheels on robot
    public static final double kWheelBase = 0.48895;
    // Distance between front and back wheels on robot
    public static final SwerveDriveKinematics kDriveKinematics =
            new SwerveDriveKinematics(
                    new Translation2d(kWheelBase / 2, kTrackWidth / 2),
                    new Translation2d(kWheelBase / 2, -kTrackWidth / 2),
                    new Translation2d(-kWheelBase / 2, kTrackWidth / 2),
                    new Translation2d(-kWheelBase / 2, -kTrackWidth / 2));

    public static final boolean kGyroReversed = false;

    // These are example values only - DO NOT USE THESE FOR YOUR OWN ROBOT!
    // These characterization values MUST be determined either experimentally or theoretically
    // for *your* robot's drive.
    // The SysId tool provides a convenient method for obtaining these values for your robot.
    public static final double ksVolts = 1;
    public static final double kvVoltSecondsPerMeter = 0.8;
    public static final double kaVoltSecondsSquaredPerMeter = 0.15;

    public static final double kMaxSpeedMetersPerSecond = 4.2;
    public static final double kMaxAngularSpeedRadiansPerSecond = 12.14;

    public static final HolonomicPathFollowerConfig kPathFollowerConfig =
            new HolonomicPathFollowerConfig(
                    new PIDConstants(1.0, 0.0, 0.0), // Translation, placeholder
                    new PIDConstants(0.2, 0.0, 0.0), // Rotation, placeholder?
                    kMaxSpeedMetersPerSecond,
                    // ToDo: remove duplicated code
                    new Translation2d(kWheelBase / 2, kTrackWidth / 2).getNorm(),
                    new ReplanningConfig()
            );
  }

  public static final class ModuleConstants {
    public static final double kMaxModuleAngularSpeedRadiansPerSecond = 8.8759 /* rps */ * 2 * Math.PI;
    public static final double kMaxModuleAngularAccelerationRadiansPerSecondSquared = 20 * 2 * Math.PI;

    public static final int kDriveEncoderCPR = 2048;
    public static final int kTurningEncoderCPR = 4096;
    public static final double kDriveReduction = 6.55;
    public static final double kWheelDiameterMeters = 4.0 * 0.0254;  // 4" OD Colson wheels
    public static final double kDriveEncoderDistancePerPulse =
            // The encoders are built into the Falcons, with a 6.55:1 reduction between the motors
            // and the wheels.
            (kWheelDiameterMeters * Math.PI) / (kDriveReduction * kDriveEncoderCPR);

    public static final double kTurningEncoderDistancePerPulse =
            // The encoders are on a 1:1 "reduction" with the module shaft.
            (2 * Math.PI) / (double) kTurningEncoderCPR;

    public static final double kFModuleTurningController = 0.281386;
    public static final double kPModuleTurningController = 4.0;

    public static final double kFModuleDriveController = 0.047;
    public static final double kPModuleDriveController = 0.05;
  }

  public static class OperatorConstants {
    public static final int kDriverControllerPort = 0;
  }

  //Wrist & Nado copied from old repo, need to be updated
  public static final class Nado {
    public static final int SPIN = 11; // CAN ID
    public static final int PINCH_CUBE = 0; // PCM channel
    public static final int PINCH_CONE = 1; // PCM channel
    public static final int PINCH = 1;
    public static final double INTAKE_CONE_V = 0.7;
    public static final double INTAKE_CUBE_V = 0.7;
    public static final double INTAKE_CURRENT_THRESHOLD = 25; // Amps
  }

  public static final class Wrist {
    public static final int WRIST = 12; // CAN ID
    public static final double WRIST_TO_END_EFFECTOR_LENGTH = 10.0;
    public static final double MAGIC_KF = 1.92;
    public static final double MAGIC_KP = 1.5;
    public static final double ENCODER_TICKS_PER_DEGREE = 4096.0 / 360.0;
    public static final double ENCODER_MAX_TICKS = 127.0 * ENCODER_TICKS_PER_DEGREE;
    public static final double ENCODER_MIN_TICKS = 0.0;
    public static final double MAGIC_CRUISE_VELOCITY = 266.0;
    public static final double MAGIC_ACCELERATION = 532.0;
    public static final double CUBE_PICKUP_ANGLE = 82.0; // degrees
    public static final double CUBE_PICKUP_ANGLE_TOLERANCE = 2.5; // degrees
    
    public static final int absoluteOrigin = 2069;
  }
}
